# Battle in the Sky - Código base para Pico Game Challenge: Spring Edition 2025
# Autor: Pablo Armando Alcaraz Valencia
# Copyright (c) 2025 Pablo Armando Alcaraz Valencia
# Licencia: MIT (ver LICENSE.txt en el paquete de distribución)
# Descripción: Este es el juego base para el concurso Pico Game Challenge. Los participantes
# deben usar este código como punto de partida, personalizándolo según las reglas del concurso.
# Requiere Raspberry Pi Pico W, pantalla OLED SSD1306, botones y buzzer configurados según la Guía Técnica.

import machine
import ssd1306
import random
import utime
import uos
import micropython
import network
import urequests
import secrets
import json


# Configuración de hardware
# Nota: En MicroPython, machine.Pin(n) se refiere al número de GPIO (GPn), no al pin físico.
# Correspondencia con los pines físicos del Raspberry Pi Pico W:
# - Pin(0)  = GP0  (pin físico 1)
# - Pin(1)  = GP1  (pin físico 2)
# - Pin(11) = GP11 (pin físico 15)
# - Pin(12) = GP12 (pin físico 16)
# - Pin(13) = GP13 (pin físico 17)
# - Pin(14) = GP14 (pin físico 19)

# Configuración de pantalla OLED
i2c = machine.I2C(0, sda=machine.Pin(0), scl=machine.Pin(1))
oled = ssd1306.SSD1306_I2C(128, 64, i2c)

# Configuración de botones
right_button = machine.Pin(11, machine.Pin.IN, machine.Pin.PULL_UP)
left_button = machine.Pin(12, machine.Pin.IN, machine.Pin.PULL_UP)
fire_button = machine.Pin(13, machine.Pin.IN, machine.Pin.PULL_UP)
buzzer = machine.PWM(machine.Pin(14))

# Dimensiones
ANCHO, ALTO = 128, 64
avatar_ancho, avatar_alto = 5, 5
obstaculo_ancho, obstaculo_alto = 10, 10

# Variables iniciales
avatar_x = (ANCHO - avatar_ancho) // 2
avatar_y = ALTO - avatar_alto - 1
avatar_velocidad = 5
obstaculos = [(random.randint(0, ANCHO - obstaculo_ancho), 0)]
obstaculo_velocidad = 4  # Velocidad inicial de los obstáculos
proyectiles = []
disparos, enemigos_destruidos, tiempo_inicio = 0, 0, utime.time()
device_id = "PABLO_ALCARAZ_{}"
num_obstaculos_max = 2

# Base de conocimientos (reglas IF-THEN)
base_de_hechos = {
    "boton_izquierda": "Mover a la izquierda",
    "boton_derecha": "Mover a la derecha",
    "boton_disparo": "Disparar proyectil",
    "colision_obstaculo": "Fin del juego",
    "proyectil_colision": "Destruir obstáculo",
    "cinco_enemigos_destruidos": "Aumentar velocidad obstáculos"
}

def inferencia(evento):
    return base_de_hechos.get(evento, "No acción")

# Función para conectarse a WiFi
def connect_to_wifi():
    wlan = network.WLAN(network.STA_IF)
    wlan.active(True)
    wlan.connect(secrets.SSID, secrets.PASSWORD)
    retry_count = 0
    max_retries = 10
    while not wlan.isconnected() and retry_count < max_retries:
        utime.sleep(1)
        retry_count += 1
    return wlan.isconnected()

# Función para enviar datos al servidor remoto
def send_data_to_server():
    if connect_to_wifi():
        url = "https://pabloalcaraz.mx/PicoChallenge/SpringEdition_2025/data/scores/hook.php"
        data = {
            "device_name": device_id,
            "disparos": disparos,
            "enemigos_destruidos": enemigos_destruidos,
            "tiempo_juego": utime.time() - tiempo_inicio
        }
        headers = {"Content-Type": "application/x-www-form-urlencoded"}
        post_data = "&".join([f"{key}={value}" for key, value in data.items()])
        try:
            response = urequests.post(url, data=post_data, headers=headers)
            print("Respuesta del servidor:", response.text)
            response.close()
        except Exception as e:
            print(f"Error al enviar datos: {e}")
    else:
        oled.fill(0)
        oled.text("No WiFi", 40, ALTO // 2 - 8)
        oled.show()
        utime.sleep(2)

# Guardar estadísticas localmente y enviar remotamente
def guardar_estadisticas():
    with open("stats.txt", "a") as f:
        tiempo_juego = utime.time() - tiempo_inicio
        f.write(f"{device_id},{disparos},{enemigos_destruidos},{tiempo_juego}\n")
    send_data_to_server()  # Enviar datos al servidor remoto

# Funciones para el buzzer
def sonido_inicio():
    buzzer.freq(523)
    buzzer.duty_u16(512)
    utime.sleep(0.2)
    buzzer.freq(392)
    utime.sleep(0.2)
    buzzer.deinit()

def sonido_disparo():
    buzzer.freq(1000)
    buzzer.duty_u16(512)
    utime.sleep(0.05)
    buzzer.deinit()

def sonido_acierto():
    buzzer.freq(440)
    buzzer.duty_u16(512)
    utime.sleep(0.1)
    buzzer.freq(523)
    utime.sleep(0.1)
    buzzer.deinit()

def sonido_fin():
    buzzer.freq(330)
    buzzer.duty_u16(512)
    utime.sleep(0.3)
    buzzer.freq(262)
    utime.sleep(0.3)
    buzzer.deinit()

def reiniciar_juego():
    global avatar_x, obstaculos, proyectiles, disparos, enemigos_destruidos, tiempo_inicio, obstaculo_velocidad
    avatar_x = (ANCHO - avatar_ancho) // 2
    obstaculos = [(random.randint(0, ANCHO - obstaculo_ancho), 0)]
    proyectiles = []
    disparos, enemigos_destruidos = 0, 0
    tiempo_inicio = utime.time()
    obstaculo_velocidad = 4  # Reiniciar velocidad de obstáculos
    oled.fill(0)
    oled.text("BattleInTheSky", 10, 16)
    oled.text("Press any button", 1, 35)
    oled.show()
    while left_button.value() == 1 and right_button.value() == 1 and fire_button.value() == 1:
        utime.sleep(0.1)
    sonido_inicio()

# Bucle principal
reiniciar_juego()
fire_prev_state = fire_button.value()
contador_iteraciones = 0
while True:
    eventos = []
    if left_button.value() == 0:
        eventos.append("boton_izquierda")
    if right_button.value() == 0:
        eventos.append("boton_derecha")
    fire_curr_state = fire_button.value()
    if fire_curr_state == 0 and fire_prev_state == 1:
        eventos.append("boton_disparo")
        disparos += 1
        sonido_disparo()
    fire_prev_state = fire_curr_state

    # Aplicar reglas de movimiento y disparo
    for evento in eventos:
        accion = inferencia(evento)
        if accion == "Mover a la izquierda":
            avatar_x = max(0, avatar_x - avatar_velocidad)
        elif accion == "Mover a la derecha":
            avatar_x = min(ANCHO - avatar_ancho, avatar_x + avatar_velocidad)
        elif accion == "Disparar proyectil":
            proyectiles.append([avatar_x + avatar_ancho // 2, avatar_y - 1])

    # Mover proyectiles y detectar colisiones
    for p in proyectiles[:]:
        p[1] -= 2
        if p[1] < 0:
            proyectiles.remove(p)
        else:
            for i, (obs_x, obs_y) in enumerate(obstaculos[:]):
                if (p[0] >= obs_x and p[0] <= obs_x + obstaculo_ancho and
                    p[1] <= obs_y + obstaculo_alto):
                    proyectiles.remove(p)
                    eventos.append("proyectil_colision")
                    enemigos_destruidos += 1
                    obstaculos[i] = (random.randint(0, ANCHO - obstaculo_ancho), 0)
                    sonido_acierto()
                    break

    # Verificar regla de velocidad después de 5 enemigos destruidos
    if enemigos_destruidos >= 5 and "cinco_enemigos_destruidos" not in eventos:
        eventos.append("cinco_enemigos_destruidos")
    for evento in eventos:
        if inferencia(evento) == "Aumentar velocidad obstáculos":
            obstaculo_velocidad = 6  # Aumentar velocidad a 6

    # Mover obstáculos
    for i, (obs_x, obs_y) in enumerate(obstaculos[:]):
        obstaculos[i] = (obs_x, obs_y + obstaculo_velocidad)
        if obstaculos[i][1] > ALTO:
            obstaculos[i] = (random.randint(0, ANCHO - obstaculo_ancho), 0)
    if len(obstaculos) < num_obstaculos_max and random.random() < 0.05:
        obstaculos.append((random.randint(0, ANCHO - obstaculo_ancho), 0))

    # Detección de colisión con avatar
    for obs_x, obs_y in obstaculos:
        if (avatar_x < obs_x + obstaculo_ancho and avatar_x + avatar_ancho > obs_x and
            avatar_y < obs_y + obstaculo_alto and avatar_y + avatar_alto > obs_y):
            eventos.append("colision_obstaculo")
            break

    # Ejecutar acciones de colisión
    for evento in eventos:
        if inferencia(evento) == "Fin del juego":
            oled.fill(0)
            oled.text("GAME OVER", 30, ALTO // 2 - 8)
            oled.show()
            guardar_estadisticas()  # Guardar localmente y enviar remotamente
            sonido_fin()
            utime.sleep(2)
            reiniciar_juego()

    # Dibujar en pantalla
    oled.fill(0)
    oled.rect(avatar_x, avatar_y, avatar_ancho, avatar_alto, 1)
    for obs_x, obs_y in obstaculos:
        oled.rect(obs_x, obs_y, obstaculo_ancho, obstaculo_alto, 1)
    for p in proyectiles:
        oled.rect(p[0], p[1], 2, 2, 1)
    oled.show()

    # Medir uso de memoria
    contador_iteraciones += 1
    if contador_iteraciones % 10 == 0:
        micropython.mem_info()

    utime.sleep(0.1)